var validationOpts = {
	errorClass: 'has-error',
	highlight: function(element, errorClass) {
		$(element).parents('.form-group').addClass(errorClass);
		if ($(element).parents('.form-group').is('.btn-group-vertical')) {
			$(element).parents('.form-group').prev().addClass(errorClass);
		}
	},
	unhighlight: function(element, errorClass) {
		if (!$(element).attr('required')
			&& Object.keys(validationOpts.rules)
			.indexOf($(element).attr('name')) < 0) return;
		$(element).parents('.form-group').removeClass(errorClass);
		if ($(element).parents('.form-group').is('.btn-group-vertical')) {
			$(element).parents('.form-group').prev()
				.removeClass(errorClass);
		}
	},
	messages: {
		'schedule-label': 'You must create a name for the schedule. '
			+ 'This will help identify which schedules are saved to '
			+ 'light poles, and what the contents of a template are.',
		'from-date': 'You must specify a start date for the schedule. '
			+ 'Start time and duration of the scheduled event are '
			+ 'optional; if not supplied, the schedule will last '
			+ 'all day.',
		'after-count': 'Occurrences cannot be less than one.'
	},
	rules: {
		'after-count': {
			min: {
				param: 1,
				depends: function(element) {
					return $(element).is(':visible');
				}
			}
		}
	},
	ignore: '.ignore',
	wrapper: 'li',
	errorContainer: '#error-list',
	errorLabelContainer: '#error-list ul'
};

$(function() {
	$('#repeats-when').change(repeatsChanged);
	$('#repeat-every').change(repeatEveryChanged);
	$('#all-day').change(allDayChanged);
	$('#repeats').change(repeatChanged);
	repeatChanged.call($('#repeats-when'));
	
	$('#schedule input,#schedule select').change(updateSummary);
	
	$('.colorpicker').ColorPickerSliders({
		hsvpanel: true,
		sliders: false,
		previewformat: 'hex',
		swatches: ['red', 'green', 'blue', 'white', 'black',
				   'yellow', 'orange', 'purple', 'pink', 'lime', 'cyan',
				   'magenta', 'olive', 'navy'],
		customswatches: 'light-colors'
	});
	$('.colorpicker + .input-group-addon').click(function() {
		$(this).prev().focus().trigger('colorpickersliders.show');
	});
	
	$('.colorpicker-brightness').ColorPickerSliders({
		flat: true,
		color: '#ffffff',
		hvspanel: false,
		sliders: true,
		previewformat: 'hex',
		swatches: false,
		connectedinput: '#intensity',
		order: {
			hsl: 1
		},
		labels: {
			hsllightness: ''
		},
		onchange: function(container, color) {
			var white = color.tiny.toRgb().r;
			$('#numeric-intensity').text((white / 255 * 100).toFixed(2) + '%');
		}
	});
	
	$('#add-color').click(addColor);
	$('.remove-color').click(removeColor);
	$('.rainbow').click(toggleRainbow);
	$('[name=light-unchanged],[name=accent-unchanged]').parent()
		.click(toggleUnchangedBtn)
		.click(function() { $(this).blur(); });
	$('#save').click(function() { saveSchedule(false); });
	$('#save-template').click(function() { saveSchedule(true); });
	
	// Populate the templates dropdown
	$.getJSON('//iot.lithl.info:1880/template', {}).done(function(data) {
		var i, k, val, keys, $option;
		
		for(i = 0; i < data.length; i++) {
			val = data[i];
			keys = Object.keys(val);
			$option = $('<option>').text(val.name);
			for(k = 0; k < keys.length; k++) {
				$option.data(keys[k], val[keys[k]]);
			}
			$('#template-selection').append($option);

			if (val.strobe) {
				(function($option) {
					$.getJSON('//iot.lithl.info:1880/template', {
						strobe: true,
						schedule_id: val.id
					}).done(function(strobeData) {
						$option.data('strobe_colors', strobeData);
					});
				})($option);
			}
		}
	});
	
	$('#template-selection').change(templateSelectionChanged);
	$('#delete-template').click(deleteSelectedTemplate);
	
	$('#schedule input,#schedule select').change(scheduleFormChanged);
	
	$('#schedule').validate(validationOpts);
});

/**
 * Deletes the currently selected schedule template
 */
function deleteSelectedTemplate() {
	var templateData = $('#template-selection').find(':selected').data();
	
	if (templateData.id) {
		$.ajax({
			url: '//iot.lithl.info:1880/template',
			type: 'DELETE',
			data: {
				id: templateData.id
			}
		}).done(function(data) {
			$('#template-selection').find(':selected').remove();
			$('#template-selection').find(':first-child')
				.prop('selected', true);
			templateSelectionChanged.call($('#template-selection'));
		});
	}
	
	$(this).blur();
}

/**
 * Reduces opacity of relevant rows when 'Do Not Change ___' buttons are
 * active
 *
 * @param event Event event object; ignored
 * @param loadedTemplate boolean whether this function is being called from
 *                               templateSelectionChanged()
 */
function toggleUnchangedBtn(event, loadedTemplate) {
	var classes = $(this).attr('class').split(' '),
		isActive = $(this).find('input').is(':checked');

	classes.splice(classes.indexOf('btn'), 1);
	classes.splice(classes.indexOf('btn-default'), 1);

	if (!loadedTemplate) isActive = !isActive;
	
	if (isActive) {
		$('.row.' + classes[0]).addClass('no-change');
	} else {
		$('.row.' + classes[0]).removeClass('no-change');
	}
}

/**
 * Saves the schedule to the database
 * 
 * @param asTemplate boolean `true` if the schedule should only be saved
 *                           as a template, `false` if it should be saved as
 *                           an actual schedule
 */
function saveSchedule(asTemplate) {
	var fields = $.serializeForm($('#schedule'), $('form input,form select'));
	
	if (!$('#schedule').valid()) return;

	fields.saveAsTemplate = asTemplate;
	fields.ids = $.urlQuery('ids');
	$.post('//iot.lithl.info:1880/template', fields).done(function(data) {
		var templateData = {},
			fieldKeys = Object.keys(fields),
			strobe = false,
			numColors = 0,
			$option = $('<option>'),
			i, isRainbow;
		
		function rgbToArray(rgb) {
			return [
				parseInt(rgb.substr(1, 2), 16) || null,
				parseInt(rgb.substr(3, 2), 16) || null,
				parseInt(rgb.substr(5, 2), 16) || null
			];
		}
		
		function getRed(rgb) { return rgbToArray(rgb)[0]; }
		function getGreen(rgb) { return rgbToArray(rgb)[1]; }
		function getBlue(rgb) { return rgbToArray(rgb)[2]; }
		
		if (asTemplate) {
			if (fieldKeys.indexOf('accent-duration-1') >= 0
				|| fields['accent-rainbow-0']) {
					strobe = true;
			}
			
			$.each(fieldKeys, function(index, k) {
				if (/accent-duration-[0-9]+/.test(k)) {
					numColors++;
				}
			});
			
			if (strobe || fields['accent-unchanged']) {
				templateData.accent_pwm_blue = null;
				templateData.accent_pwm_green = null;
				templateData.accent_pwm_red = null;
			} else {
				templateData.accent_pwm_blue = 
					getBlue(fields['accent-color-0']);
				templateData.accent_pwm_green = 
					getGreen(fields['accent-color-0']);
				templateData.accent_pwm_red = 
					getRed(fields['accent-color-0']);
			}
			
			templateData.end_after_num = parseInt(fields['after-count'])
				|| null;
			
			try { templateData.end_date = fields['to-date'].toISOString(); }
			catch (e) { templateData.end_date = null; }
			try {
				templateData.end_time = fields['to-time']
					.toLocaleTimeString('en-US', { hour12: false });
			} catch (e) { templateData.end_time = null; }

			try { templateData.end_on = fields['on-date'].toISOString(); }
			catch (e) { templateData.end_on = null; }

			templateData.id = data.insertId;
			
			if (fields['light-unchanged']) {
				templateData.light_pwm = null;
			} else {
				templateData.light_pwm = getRed(fields.intensity);
			}

			templateData.name = fields['schedule-label'];
			$option.text(templateData.name);
			
			templateData.repeat_by = parseInt(fields['repeat-by']) || null;
			templateData.repeat_every =
				parseInt(fields['repeat-every']) || null;
			templateData.repeat_sunday = fields['repeat-on-sun'] ? 1 : 0;
			templateData.repeat_monday = fields['repeat-on-mon'] ? 1 : 0;
			templateData.repeat_tuesday = fields['repeat-on-tue'] ? 1 : 0;
			templateData.repeat_wednesday = fields['repeat-on-wed'] ? 1 : 0;
			templateData.repeat_thursday = fields['repeat-on-thu'] ? 1 : 0;
			templateData.repeat_friday = fields['repeat-on-fri'] ? 1 : 0;
			templateData.repeat_saturday = fields['repeat-on-sat'] ? 1 : 0;
			
			if (fields['repeats']) {
				templateData.repeat_type = fields['repeat-when'];
			} else {
				templateData.repeat_type = null;
			}

			templateData.start_date = fields['from-date'].toISOString();
			try {
				templateData.start_time = fields['from-time']
					.toLocaleTimeString('en-US', { hour12: false });
			} catch (e) { templateData.start_time = null; }

			templateData.strobe = strobe ? 1 : 0;
			
			if (strobe) {
				templateData.strobe_colors = [];
				for (i = 0; i < numColors; i++) {
					isRainbow = fields['accent-rainbow-' + i];
					templateData.strobe_colors.push({
						accent_pwm_blue: isRainbow ? null
							: getBlue(fields['accent-color-' + i]),
						accent_pwm_green: isRainbow ? null
							: getGreen(fields['accent-color-' + i]),
						accent_pwm_red: isRainbow ? null
							: getRed(fields['accent-color-' + i]),
						duration: (fields['accent-duration-' + i].getHours()
							 * 60 * 60) + (fields['accent-duration-' + i]
							 .getMinutes() * 60)
							 + fields['accent-duration-' + i].getSeconds(),
						rainbow: isRainbow ? 1 : 0
					});
				}
			}
			
			$.each(templateData, function(key, val) {
				$option.data(key, val);
			});

			$('#template-selection').append($option);
			$option.prop('selected', true);
			templateSelectionChanged.call($('#template-selection'));
			$('#schedule-label').val('');
		}
	});
}

/**
 * When the schedule form changes to something other than the template,
 * change the template dropdown to not indicate that template.
 */
function scheduleFormChanged() {
	$('#template-selection').val('');
}

/**
 * Fill the schedule form with template values when a template is chosen
 */
function templateSelectionChanged() {
	var templateInfo = $(this).find(':selected').data(),
		bluePWM = parseInt(templateInfo.accent_pwm_blue),
		blue = (isNaN(bluePWM) ? 163 : bluePWM).toString(16),
		greenPWM = parseInt(templateInfo.accent_pwm_green),
		green = (isNaN(greenPWM) ? 139 : greenPWM).toString(16),
		redPWM = parseInt(templateInfo.accent_pwm_red),
		red = (isNaN(redPWM) ? 218 : redPWM).toString(16),
		endAfter = parseInt(templateInfo.end_after_num),
		endDate = new Date(templateInfo.end_date),
		endOn = new Date(templateInfo.end_on),
		endTime = parseTimeString(templateInfo.end_time),
		id = templateInfo.id,
		lightPWM = parseInt(templateInfo.light_pwm),
		intensity = '#'+(lightPWM || 255).toString(16).repeat(3),
		repeatBy = templateInfo.repeat_by || 1,
		repeatEvery = templateInfo.repeat_every,
		repeatDays = {
			sun: !!templateInfo.repeat_sunday,
			mon: !!templateInfo.repeat_monday,
			tue: !!templateInfo.repeat_tuesday,
			wed: !!templateInfo.repeat_wednesday,
			thu: !!templateInfo.repeat_thursday,
			fri: !!templateInfo.repeat_friday,
			sat: !!templateInfo.repeat_saturday
		},
		repeatType = parseInt(templateInfo.repeat_type),
		startDate = new Date(templateInfo.start_date),
		startTime = parseTimeString(templateInfo.start_time),
		strobe = !!templateInfo.strobe,
		strobeColors = templateInfo.strobe_colors;
		
	$(this).blur();
	
	// Format colors			
	red = red.length < 2 ? '0' + red : red;
	green = green.length < 2 ? '0' + green : green;
	blue = blue.length < 2 ? '0' + blue : blue;
	
	$('#from-date').parent().data('datetimepicker').setDate(startDate);
	$('[name=from-time]').parent().data('datetimepicker')
		.setLocalDate(startTime);
	$('#to-date').parent().data('datetimepicker').setDate(endDate);
	$('[name=to-time]').parent().data('datetimepicker')
		.setLocalDate(endTime);
	if (!startTime && !endTime) {
		$('#all-day').prop('checked', true).parent().addClass('active');
	} else {
		$('#all-day').prop('checked', false).parent().removeClass('active');
	}
	if (isNaN(repeatType)) {
		$('#repeats').prop('checked', false).parent().removeClass('active');
	} else {
		$('#repeats').prop('checked', true).parent().addClass('active');
		if (repeatType === 2) {
			if (!repeatDays.sun && repeatDays.mon && repeatDays.tue
				&& repeatDays.wed && repeatDays.thu && repeatDays.fri
				&& !repeatDays.sat) {console.log('weekdays');
					$('#repeats-when').val(5);
			} else if (!repeatDays.sun && repeatDays.mon && !repeatDays.tue
				&& repeatDays.wed && !repeatDays.thu && repeatDays.fri
				&& !repeatDays.sat) {
					$('#repeats-when').val(6);
			} else if (!repeatDays.sun && !repeatDays.mon && repeatDays.tue
				&& !repeatDays.wed && repeatDays.thu && !repeatDays.fri
				&& !repeatDays.sat) {
					$('#repeats-when').val(7);
			} else {
				$('#repeats-when').val(repeatType);
			}
		}
		$('#repeat-every').val(repeatEvery);
		$.each(repeatDays, function(day, val) {
			if (val) $('#' + day).parent().addClass('active');
			else $('#' + day).parent().removeClass('active');
			$('#' + day).prop('checked', val);
		});
		$('[name=repeat-by]').prop('checked', false)
			.parent().removeClass('active');
		$('[name=repeat-by][value=' + repeatBy + ']').prop('checked', true)
			.parent().addClass('active');
		$('[name=end-repeat][value=1]').prop('checked', true);
		if (isNaN(endAfter)) {
			$('[name=after-count]').val('');
		} else {
			$('[name=after-count]').val(endAfter);
			$('[name=end-repeat][value=2]').prop('checked', true);
		}
		if (endOn) {
			$('[name=on-date]').parent()
				.data('datetimepicker').setDate(endOn);
			$('[name=end-repeat][value=3]').prop('checked', true);
		} else {
			$('[name=on-date]').val('');
		}
		$('[name=end-repeat]:checked').parent().addClass('active');
		$('[name=end-repeat]:not(:checked)').parent().removeClass('active');
	}
	repeatChanged.call($('#repeats'));
	allDayChanged.call($('#all-day'));
	updateSummary();

	$('#intensity + div').trigger('colorpickersliders.updateColor',intensity);
	
	if (strobe) {
		// Check strobe_colors table for rainbow, extra colors, and time
		$('[name^=accent-color-]:not([name$=-0])').parents('.accent-colors')
			.remove();
		$.each(strobeColors, function(idx, data) {
			var redPWM = parseInt(data.accent_pwm_red),
				greenPWM = parseInt(data.accent_pwm_green),
				bluePWM = parseInt(data.accent_pwm_blue),
				red = (redPWM || 218).toString(16),
				green = (greenPWM || 139).toString(16),
				blue = (bluePWM || 163).toString(16),
				duration = new Date(0, 0, 0, 0, 0, 0, data.duration * 1000),
				rainbow = !!data.rainbow;
			
			red = red.length < 2 ? '0' + red : red;
			green = green.length < 2 ? '0' + green : green;
			blue = blue.length < 2 ? '0' + blue : blue;
			
			if (idx > 0) {
				addColor.call($('#add-color'));
			}
			$('[name=accent-duration-' + idx + ']').parent()
				.data('datetimepicker').setLocalDate(duration);
			if (rainbow) {
				$('[name=accent-rainbow-' + idx + ']').prop('checked', true)
					.parent().addClass('active');
				disableColor($('[name=accent-color-' + idx + ']'));
			} else {
				$('[name=accent-rainbow-' + idx + ']').prop('checked', false)
					.parent().removeClass('active');
				enableColor($('[name=accent-color-' + idx + ']'));
				$('[name=accent-color-' + idx + ']')
					.trigger('colorpickersliders.updateColor',
						'#' + red + green + blue);
			}
		});
	} else {
		// We only have one color, and it's not rainbow
		$('[name=accent-color-0]').trigger('colorpickersliders.updateColor',
			'#' + red + green + blue);
		$('[name^=accent-color-]:not([name$=-0])').parents('.accent-colors')
			.remove();
	}
	if (isNaN(lightPWM)) {
		$('[name=light-unchanged]').prop('checked', true)
			.parent().addClass('active');
	} else {
		$('[name=light-unchanged]').prop('checked', false)
			.parent().removeClass('active');
	}
	toggleUnchangedBtn.call($('[name=light-unchanged]').parent(), null, true);
	
	if (!strobe && isNaN(redPWM) && isNaN(greenPWM) && isNaN(bluePWM)) {
		$('[name=accent-unchagned]').prop('checked', true)
			.parent().addClass('active');
	} else {
		$('[name=accent-unchanged]').prop('checked', false)
			.parent().removeClass('active');
	}
	toggleUnchangedBtn.call($('[name=accent-unchanged]').parent(), null, true);
}

/**
 * Parses a string in the format 'hh:mm:ss'
 *
 * @param time String a string representation of the time
 * @param pm boolean if true and the hours portion of the time string is
 *                   less than 12, 12 hours will be added to the time (making
 *                   it a 24hr representation of a PM time)
 * @return a Date object on December 31, 1969 with the appropriate time.
 *         `null` will be returned if `time` is falsey
 */
function parseTimeString(time, pm) {
	var result = new Date(null),
		parts = (time || '').split(':'),
		hour = parseInt(parts[0]),
		minute = parseInt(parts[1])
		second = parseInt(parts[2]);
	
	if (!time) return null;
	
	if (pm && hour < 12) hour += 12;
	result.setHours(hour, minute, second);
	return result;
}

/**
 * Toggle the state of the Bootstrap button to match the underlying checkbox.
 *
 * When the rainbow checkbox is checked, the colorpicker in the same row is
 * disabled. The color that was set to that colorpicker is saved in a
 * data attribute so that it can be restored if the rainbow checkbox is
 * unchecked later.
 */
function toggleRainbow() {
	var $target = $(this),
		rainbowChecked = $target.find('input').is(':checked'),
		$color = $target.parent().prev().find('.colorpicker');
	
	if (!rainbowChecked) {
		disableColor($color);
	} else {
		enableColor($color);
	}
}

/**
 * Disable an accent color colorpicker
 *
 * @param $color jQuery the DOM element representing the colorpicker
 */
function disableColor($color) {
	$color.data('tmp-storage', $color.val())
		  .trigger('colorpickersliders.updateColor', '#eee')
		  .val('')
		  .attr('disabled', true)
		  .next().off('click');
}

/**
 * Enable an accent color colorpicker
 *
 * @param $color jQuery the DOM element representing the colorpicker
 */
function enableColor($color) {
	if (!$color.attr('disabled')) return;
	
	$color.attr('disabled', false)
		  .val($color.data('tmp-storage'))
		  .trigger('colorpickersliders.updateColor', $color.val())
		  .next().click(function() {
		$(this).prev().focus().trigger('colorpickersliders.show');
	});
}

/**
 * Delete an accent color
 */
function removeColor() {
	var $row = $(this).parent().parent(),
		$colors, $rainbows, $timespans;
	
	$row.remove();

	$colors = $('.colorpicker');
	$rainbows = $('.rainbow');
	$timespans = $('.timespan');
	$colors.each(function(idx, color) {
		$(color).attr('name', 'accent-color-' + idx);
	});
	$rainbows.each(function(idx, rainbow) {
		$(rainbow).find('input').attr('name', 'accent-rainbow-' + idx);
	});
	$timespans.each(function(idx, timespan) {
		$(timespan).find('input').attr('name', 'accent-duration-' + idx);
	});
}

/**
 * Add an accent color
 */
function addColor() {
	var $row = $(this).parent().parent().prev(),
		$newRow,
		numColors = $('.colorpicker').length;
	
	$newRow = $row.clone();
	$newRow.find('#accent-color-label').remove();
	$newRow.find('.colorpicker').ColorPickerSliders({
		hsvpanel: true,
		sliders: false,
		previewformat: 'hex',
		swatches: ['red', 'green', 'blue', 'white', 'black',
					'yellow', 'orange', 'purple', 'pink', 'lime', 'cyan',
					'magenta', 'olive', 'navy'],
		customswatches: 'light-colors'
	}).next('.input-group-addon').click(function() {
		$(this).prev().focus().trigger('colorpickersliders.show');
	}).prev().attr('name', 'accent-color-' + numColors).removeAttr('disabled');
	$newRow.find('.color-duration > label').remove();
	$newRow.find('.timespan').datetimepicker({
		format: 'hh:mm:ss',
		pickDate: false
	}).find('input').attr('name', 'accent-duration-' + numColors);
	$newRow.find('.rainbow').click(toggleRainbow)
		   .removeClass('active')
		   .parent().css('padding-top', 0)
		   .find('input').attr('name', 'accent-rainbow-' + numColors)
		   				 .prop('checked', false);
	$newRow.find('.actions').removeAttr('style')
							 .find('.remove-color').click(removeColor);
	$row.after($newRow);

	// Fix up/down arrows in time pickers	
	$('i.icon-chevron-up').addClass('glyphicon glyphicon-chevron-up')
		.removeClass('icon-chevron-up')
		.parent().addClass('btn-default');
	$('i.icon-chevron-down').addClass('glyphicon glyphicon-chevron-down')
		.removeClass('icon-chevron-down')
		.parent().addClass('btn-default');
}

/**
 * Build the summary text at the bottom of the 'repeat...' section
 */
function updateSummary() {
	var data = $.serializeForm($('#schedule'),
						$('.repeat-info input,.repeat-info select')),
		$summary = $('#summary'),
		summary,
		days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday',
			'Thursday', 'Friday', 'Saturday'],
		months = ['January', 'February', 'March', 'April',
					'May', 'June', 'July', 'August',
					'September', 'October', 'November', 'December'],
		prefixes = ['first', 'second', 'third', 'fourth', 'fifth'];
	
	if (data['from-date']) {
		data['from-date'] = new Date(data['from-date']);
	}
	if (data['to-date']) {
		data['to-date'] = new Date(data['to-date']);
	}
	if (data['on-date'] ){
		data['on-date'] = new Date(data['on-date']);
	}
	if (data['repeat-when'] === 5) {
		data['repeat-on-sun'] = false;
		data['repeat-on-mon'] = true;
		data['repeat-on-tue'] = true;
		data['repeat-on-wed'] = true;
		data['repeat-on-thu'] = true;
		data['repeat-on-fri'] = true;
		data['repeat-on-sat'] = false;
	}
	if (data['repeat-when'] === 6) {
		data['repeat-on-sun'] = false;
		data['repeat-on-mon'] = true;
		data['repeat-on-tue'] = false;
		data['repeat-on-wed'] = true;
		data['repeat-on-thu'] = false;
		data['repeat-on-fri'] = true;
		data['repeat-on-sat'] = false;
	}
	if (data['repeat-when'] === 7) {
		data['repeat-on-sun'] = false;
		data['repeat-on-mon'] = false;
		data['repeat-on-tue'] = true;
		data['repeat-on-wed'] = false;
		data['repeat-on-thu'] = true;
		data['repeat-on-fri'] = false;
		data['repeat-on-sat'] = false;
	}
	switch (data['repeat-when']) {
		case 1: // Daily
			switch (data['repeat-every']) {
				case 1:
					summary = 'Daily';
					break;
				default:
					summary = 'Every ' + data['repeat-every'] + ' days';
					break;
			}
			break;
		case 5: // weekdays
		case 6: // MWF
		case 7: // TR
		case 2: // Weekly
			if (data['repeat-every'] === 1 || data['repeat-when'] !== 2) {
				summary = 'Weekly on ';
			} else {
				summary = 'Every ' + data['repeat-every'] + ' weeks on ';
			}
			
			if (!data['repeat-on-sun'] && data['repeat-on-mon']
				&& data['repeat-on-tue'] && data['repeat-on-wed']
				&& data['repeat-on-thu'] && data['repeat-on-fri']
				&& !data['repeat-on-sat']) {
				summary += 'weekdays';
			} else if (!(data['repeat-on-sun'] || data['repeat-on-mon']
						|| data['repeat-on-tue'] || data['repeat-on-wed']
						|| data['repeat-on-thu'] || data['repeat-on-fri']
						|| data['repeat-on-sat'])) {
				summary += 'Monday';
			} else if (data['repeat-on-sun'] && data['repeat-on-mon']
						&& data['repeat-on-tue'] && data['repeat-on-wed']
						&& data['repeat-on-thu'] && data['repeat-on-fri']
						&& data['repeat-on-sat']) {
				summary += 'all days';
			} else {
				if (data['repeat-on-sun']) {
					summary += 'Sunday, ';
				}
				if (data['repeat-on-mon']) {
					summary += 'Monday, ';
				}
				if (data['repeat-on-tue']) {
					summary += 'Tuesday, ';
				}
				if (data['repeat-on-wed']) {
					summary += 'Wednesday, ';
				}
				if (data['repeat-on-thu']) {
					summary += 'Thursday, ';
				}
				if (data['repeat-on-fri']) {
					summary += 'Friday, ';
				}
				if (data['repeat-on-sat']) {
					summary += 'Saturday, ';
				}
				summary = summary.substring(0, summary.lastIndexOf(','));
			}
			break;
		case 3: // Monthly
			switch (data['repeat-every']) {
				case 1:
					summary = 'Monthly on ';
					break;
				default:
					summary = 'Every ' + data['repeat-every'] + ' months on ';
					break;
			}
			
			switch (data['repeat-by']) {
				case 1:
					summary += 'day ';
					if (data['from-date']) {
						summary += data['from-date'].getDate();
					} else {
						summary += '1';
					}
					break;
				case 2:
					summary += 'the ';
					
					if (data['from-date']) {
						summary += prefixes[0|data['from-date'].getDate() / 7];
						summary += ' ' + days[data['from-date'].getDay()];
					} else {
						summary += 'first Monday';
					}
					break;
			}
			break;
		case 4: // Yearly
			switch (data['repeat-every']) {
				case 1:
					summary = 'Annually on ';
					break;
				default:
					summary = 'Every ' + data['repeat-every'] + ' years on ';
					break;
			}
			
			if (data['from-date']) {
				summary += months[data['from-date'].getMonth()] + ' ';
				summary += data['from-date'].getDate();
			} else {
				summary += 'January 1';
			}
			break;
	}
	
	switch (data['end-repeat']) {
		case 2: // After X occurrences
			if (data['after-count']) {
				summary += ', ' + data['after-count'] + ' time';
				if (data['after-count'] > 1) {
					summary += 's';
				}
			}
			break;
		case 3: // On MM/DD/YYYY
			if (data['on-date']) {
				summary += ', until ';
				summary += data['on-date'].toDateString();
			}
			break;
	}
	$summary.text(summary);
}

/**
 * Show/hide the entire 'repeats...' section
 */
function repeatChanged() {
	var $repeat = $(this);
	
	if ($repeat.is(':checked')) {
		$('.repeat-info').show();
		repeatsChanged.call($('#repeats-when'));
	} else {
		$('.repeat-info').hide();
	}
	$('#schedule').validate(validationOpts).element($('#after-count'));
	//$('#after-count').valid();
}

/**
 * Show/hide the from and to time pickers
 */
function allDayChanged() {
	var allDay = $(this).is(':checked');
	
	if (allDay) {
		$('.time').hide();
		$('.date').removeClass('group-half');
	} else {
		$('.time').show();
		$('.date').addClass('group-half');
	}
}

/**
 * Hide or show parts of the 'repeat...' section of the schedule
 */
function repeatsChanged() {
	var $toggleBlocks = $('[data-repeats]'),
		selectValue = $(this).val(),
		$selectUnitOption = $(this).find(':selected'),
		selectUnitSingular = $selectUnitOption.data('unit-singular'),
		selectUnitPlural = $selectUnitOption.data('unit-plural'),
		selectId = parseInt(selectValue);
	
	$toggleBlocks.hide().each(function(idx, element) {
		var $e = $(element),
			repData = $e.data('repeats');
		
		$(repData).each(function(x, id) { id === selectId && $e.show(); });
	});
	
	if ($selectUnitOption) {
		$('#repeat-every-unit').data('singular', selectUnitSingular)
							   .data('plural', selectUnitPlural);
		repeatEveryChanged();
	}
}

/**
 * Update the text after the 'repeat'every' dropdown
 */
function repeatEveryChanged() {
	var $select = $('#repeat-every'),
		$unit = $('#repeat-every-unit');

	if (parseInt($select.val()) === 1) {
		$unit.text($unit.data('singular'));
	} else {
		$unit.text($unit.data('plural'));
	}
}
